# Reading blockchain

Reading blockchain state only requires an instance of `JsonRpc` connected to a node.

```javascript
const { JsonRpc } = require('eosjs');
const fetch = require('node-fetch');           // node only; not needed in browsers
const rpc = new JsonRpc('http://localhost:8888', { fetch });
```

## Examples

### Get table rows

Get the first 10 token balances of account _testacc_.

```javascript
const resp = await rpc.get_table_rows({
    json: true,              // Get the response as json
    code: 'eosio.token',     // Contract that we target
    scope: 'testacc',        // Account that owns the data
    table: 'accounts',       // Table name
    limit: 10,               // Maximum number of rows that we want to get
    reverse: false,          // Optional: Get reversed data
    show_payer: false,       // Optional: Show ram payer
});

console.log(resp.rows);
```
Output:

```json
{
  "rows": [{
      "balance": "100.0000 HAK"
    }
  ],
  "more": false
}
```

### Get one row by index

```javascript
const resp = await rpc.get_table_rows({
    json: true,                 // Get the response as json
    code: 'contract',           // Contract that we target
    scope: 'contract',          // Account that owns the data
    table: 'profiles',          // Table name
    lower_bound: 'testacc',     // Table primary key value
    limit: 1,                   // Here we limit to 1 to get only the
    reverse: false,             // Optional: Get reversed data
    show_payer: false,          // Optional: Show ram payer
});
console.log(resp.rows);
```
Output:

```json
{
  "rows": [{
      "user": "testacc",
      "age": 21,
      "surname": "Martin"
    }
  ],
  "more": false
}
```

### Get one row by secondary index

```javascript
const resp = await rpc.get_table_rows({
    json: true,                 // Get the response as json
    code: 'contract',           // Contract that we target
    scope: 'contract',          // Account that owns the data
    table: 'profiles',          // Table name
    table_key: 'age',           // Table secondaray key name
    lower_bound: 21,            // Table secondary key value
    limit: 1,                   // Here we limit to 1 to get only row
    reverse: false,             // Optional: Get reversed data
    show_payer: false,          // Optional: Show ram payer
});
console.log(resp.rows);
```
Output:

```json
{
  "rows": [{
      "user": "testacc",
      "age": 21,
      "surname": "Martin"
    }
  ],
  "more": false
}
```

### Get currency balance

```javascript
console.log(await rpc.get_currency_balance('eosio.token', 'testacc', 'HAK'));
```
Output:

```json
[ "1000000000.0000 HAK" ]
```

### Get account info

```javascript
console.log(await rpc.get_account('testacc'));
```
Output:

```json
{ "account_name": "testacc",
  "head_block_num": 1079,
  "head_block_time": "2018-11-10T00:45:53.500",
  "privileged": false,
  "last_code_update": "1970-01-01T00:00:00.000",
  "created": "2018-11-10T00:37:05.000",
  "ram_quota": -1,
  "net_weight": -1,
  "cpu_weight": -1,
  "net_limit": { "used": -1, "available": -1, "max": -1 },
  "cpu_limit": { "used": -1, "available": -1, "max": -1 },
  "ram_usage": 2724,
  "permissions":
   [ { "perm_name": "active", "parent": "owner", "required_auth": [] },
     { "perm_name": "owner", "parent": "", "required_auth": [] } ],
  "total_resources": null,
  "self_delegated_bandwidth": null,
  "refund_request": null,
  "voter_info": null }
```

### Get block

```javascript
console.log(await rpc.get_block(1));
```
Output:

```json
{ "timestamp": "2018-06-01T12:00:00.000",
  "producer": "",
  "confirmed": 1,
  "previous": "0000000000000000000000000000000000000000000000000000000000000000",
  "transaction_mroot": "0000000000000000000000000000000000000000000000000000000000000000",
  "action_mroot": "cf057bbfb72640471fd910bcb67639c22df9f92470936cddc1ade0e2f2e7dc4f",
  "schedule_version": 0,
  "new_producers": null,
  "header_extensions": [],
  "producer_signature": "SIG_K1_111111111111111111111111111111111111111111111111111111111111111116uk5ne",
  "transactions": [],
  "block_extensions": [],
  "id": "00000001bcf2f448225d099685f14da76803028926af04d2607eafcf609c265c",
  "block_num": 1,
  "ref_block_prefix": 2517196066 }
```
